﻿/*
    Sunburst: Halo Reach Game Variant Rehasher

    Copyright (c) Nuklear

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

using System;
using System.IO;
using System.Diagnostics;

namespace Sunburst.Lite
{
    public static class GameVariant
    {
        public static readonly byte[] ReachKey = new byte[] {
            0xED, 0xD4, 0x30, 0x09, 0x66, 0x6D, 0x5C, 0x4A, 0x5C, 0x36, 0x57,
            0xFA, 0xB4, 0x0E, 0x02, 0x2F, 0x53, 0x5A, 0xC6, 0xC9, 0xEE, 0x47,
            0x1F, 0x01, 0xF1, 0xA4, 0x47, 0x56, 0xB7, 0x71, 0x4F, 0x1C, 0x36, 
            0xEC
        };

        /// <summary>
        /// Rehashes a Halo: Reach game variant.
        /// </summary>
        /// <param name="input">The input file to rehash</param>
        /// <param name="output">The output file to save the rehashed data to</param>
        /// <returns>The path to the output file</returns>
        public static void FixHash(string input, string output)
        {
            byte[] buffer;
            using (var fileStream = File.OpenRead(input))
            {
                using (var reader = new BinaryReader(fileStream))
                {
                    // Get the length of the data to be hashed.
                    reader.BaseStream.Seek(0xD314, SeekOrigin.Begin);
                    var lengthData = BitConverter.GetBytes(reader.ReadInt32());
                    Array.Reverse(lengthData);
                    var length = BitConverter.ToInt32(lengthData, 0);

                    // Copy the data to be hashed to a buffer.
                    buffer = new byte[length + 4];
                    reader.BaseStream.Seek(0xD314, SeekOrigin.Begin);
                    reader.Read(buffer, 0, buffer.Length);
                }
            }

            // Compute the hash.
            var hasher = new SaltedSHA1(ReachKey);
            hasher.TransformFinalBlock(buffer, 0, buffer.Length);

            // Write the hash back to the file.
            using (var outputStream = File.OpenWrite(output))
            {
                using (var writer = new BinaryWriter(outputStream))
                {
                    writer.Seek(0xD2FC, SeekOrigin.Begin);
                    writer.Write(hasher.Hash);
                }
            }

            Debug.WriteLine(BitConverter.ToString(hasher.Hash));
        }
    }
}
